<?php
declare(strict_types=1);

namespace OncoAmazonPay;


use AmazonPayApiSdkExtension\Struct\AddressDetails;
use AmazonPayApiSdkExtension\Struct\AddressRestrictions;
use AmazonPayApiSdkExtension\Struct\CheckoutSession;
use AmazonPayApiSdkExtension\Struct\DeliverySpecifications;
use AmazonPayApiSdkExtension\Struct\MerchantMetadata;
use AmazonPayApiSdkExtension\Struct\PaymentDetails;
use AmazonPayApiSdkExtension\Struct\Price;
use AmazonPayApiSdkExtension\Struct\WebCheckoutDetails;
use Exception;
use order_ORIGIN;

class CheckoutService
{


    /**
     * @var ConfigurationService
     */
    private $configurationService;
    /**
     * @var LogService
     */
    private $logService;
    /**
     * @var ApiService
     */
    private $apiService;

    public function __construct()
    {
        $this->apiService = new ApiService();
        $this->configurationService = new ConfigurationService();
        $this->logService = new LogService();
    }

    /**
     * @return CheckoutSession|null
     * @throws Exception
     */
    public function createCheckoutSession($forceSandbox = false)
    {
        $checkoutSession = $this->createCheckoutSessionBaseObject();
        return $this->apiService->getClient($forceSandbox)->createCheckoutSession($checkoutSession);

    }

    public function createCheckoutSessionBaseObject(): CheckoutSession
    {
        $storeName = STORE_NAME;
        $encoding = mb_detect_encoding($storeName, ['UTF-8', 'ISO-8859-1', 'ISO-8859-2', 'ISO-8859-15']);
        if ($encoding !== 'UTF-8') {
            $storeName = mb_convert_encoding($storeName, 'UTF-8', $encoding);
        }
        $storeName = (mb_strlen($storeName) <= 50) ? $storeName : (mb_substr($storeName, 0, 47) . '...');

        $merchantData = new MerchantMetadata();
        $merchantData->setMerchantStoreName($storeName);
        $merchantData->setCustomInformation($this->configurationService->getCustomInformationString());

        $webCheckoutDetails = new WebCheckoutDetails();
        $webCheckoutDetails->setCheckoutReviewReturnUrl($this->configurationService->getReviewReturnUrl());

        $addressRestrictions = new AddressRestrictions();
        $addressRestrictions->setType('Allowed')
            ->setRestrictions($this->configurationService->getAllowedCountries());
        $deliverySpecifications = new DeliverySpecifications();
        $deliverySpecifications->setAddressRestrictions($addressRestrictions);

        $checkoutSession = new CheckoutSession();
        $checkoutSession->setMerchantMetadata($merchantData)
            ->setWebCheckoutDetails($webCheckoutDetails)
            ->setStoreId($this->configurationService->getConfiguration()->getClientId())
            ->setPlatformId($this->configurationService->getPlatformId())
            ->setDeliverySpecifications($deliverySpecifications);
        return $checkoutSession;
    }

    /**
     * @param order_ORIGIN $order
     * @return CheckoutSession
     */
    public function createCheckoutSessionObjectForPurePayment($order)
    {
        $checkoutSession = $this->createCheckoutSessionBaseObject();
        $checkoutSession->getWebCheckoutDetails()
            ->setCheckoutResultReturnUrl($this->configurationService->getCheckoutResultReturnUrl())
            ->setCheckoutReviewReturnUrl(null)
            ->setCheckoutCancelUrl($this->configurationService->getCancelUrl())
            ->setCheckoutMode(WebCheckoutDetails::CHECKOUT_MODE_PROCESS_ORDER);

        $paymentDetails = (new PaymentDetails())
            ->setPaymentIntent('Authorize')
            ->setCanHandlePendingAuthorization($this->configurationService->getConfiguration()->getCanHandlePendingAuthorization())
            ->setChargeAmount(new Price(['amount' => $order->info['total'], 'currencyCode' => $order->info['currency']]))
            ->setPresentmentCurrency($order->info['currency']);

        $checkoutSession->setPaymentDetails($paymentDetails);
        $checkoutSession->setAddressDetails($this->getAddressDetails());
        return $checkoutSession;

    }

    /**
     * @param order_ORIGIN $order
     * @return mixed
     */
    protected function getAddressDetails()
    {
        $shippingAddress = DbAdapter::fetch("
            SELECT 
                ab.entry_firstname, 
                ab.entry_lastname, 
                ab.entry_company, 
                ab.entry_street_address, 
                ab.entry_house_number, 
                ab.entry_additional_info, 
                ab.entry_suburb, 
                ab.entry_postcode, 
                ab.entry_city,
                c.countries_iso_code_2, 
                ab.entry_state 
            FROM 
                " . TABLE_ADDRESS_BOOK . " ab 
                LEFT JOIN " . TABLE_COUNTRIES . " c ON (ab.entry_country_id = c.countries_id) 
            WHERE 
                ab.customers_id = ? 
                    AND 
                ab.address_book_id = ?
            ", [$_SESSION['customer_id'], $_SESSION['sendto']]);

        return (new AddressDetails())
            ->setName($shippingAddress['entry_firstname'] . ' ' . $shippingAddress['entry_lastname'])
            ->setAddressLine1($shippingAddress['entry_company'] ?: $shippingAddress['entry_street_address'])
            ->setAddressLine2($shippingAddress['entry_company'] ? $shippingAddress['entry_street_address'] : $shippingAddress['entry_additional_info'])
            ->setAddressLine3($shippingAddress['entry_company'] ? $shippingAddress['entry_additional_info'] : $shippingAddress['entry_suburb'])
            ->setPostalCode($shippingAddress['entry_postcode'])
            ->setCity($shippingAddress['entry_city'])
            ->setCountryCode($shippingAddress['countries_iso_code_2'])
            ->setPhoneNumber('00000');
    }

    /**
     * @param string $checkoutSessionId
     * @return CheckoutSession|null
     */
    public function getCheckoutSession(string $checkoutSessionId)
    {
        try {
            return $this->apiService->getClient()->getCheckoutSession($checkoutSessionId);
        } catch (Exception $e) {
            $this->logService->error('getCheckoutSession failed', ['msg' => $e->getMessage()]);
        }
        return null;
    }

    /**
     * @param string $checkoutSessionId
     * @param CheckoutSession $checkoutSession
     * @return CheckoutSession|null
     */
    public function updateCheckoutSession(string $checkoutSessionId, CheckoutSession $checkoutSession)
    {
        try {
            return $this->apiService->getClient()->updateCheckoutSession($checkoutSessionId, $checkoutSession);
        } catch (Exception $e) {
            $this->logService->error('updateCheckoutSession failed', ['msg' => $e->getMessage()]);
        }
        return null;
    }

    public function setOrderIdToChargePermission($chargePermissionId, $orderId)
    {

        $this->apiService->getClient()->updateChargePermission(
            $chargePermissionId,
            ['merchantMetadata' => ['merchantReferenceId' => $orderId]]
        );
    }
}
